<?php
/**
 * Benz Service API
 * سرویس Proxy برای Doppy Content Generator
 */

// ============================================================================
// تنظیمات اولیه
// ============================================================================

header('Content-Type: application/json; charset=utf-8');
header('Access-Control-Allow-Origin: *');
header('Access-Control-Allow-Methods: GET, POST, OPTIONS');
header('Access-Control-Allow-Headers: Content-Type, Authorization, X-Plugin-Version, X-Site-URL');

if ($_SERVER['REQUEST_METHOD'] === 'OPTIONS') {
    http_response_code(200);
    exit;
}

// ============================================================================
// کنفیگ
// ============================================================================

define('BENZ_API_VERSION', '1.0.0');
define('BENZ_LOG_FILE', __DIR__ . '/logs/benz.log');
define('BENZ_CACHE_DIR', __DIR__ . '/cache/');

// ایجاد پوشه‌های مورد نیاز
@mkdir(__DIR__ . '/logs', 0755, true);
@mkdir(BENZ_CACHE_DIR, 0755, true);

// API Keys مجاز (تغییر دهید!)
$VALID_API_KEYS = [
    'your-secret-api-key-here' => [
        'name' => 'Default Client',
        'rate_limit' => 1000,
        'domains' => ['*']
    ]
];

// Google Gemini Settings
$GEMINI_API_URL = 'https://generativelanguage.googleapis.com/v1beta/models';
$GEMINI_API_KEY = 'YOUR_GEMINI_API_KEY'; // تغییر دهید!

// Doppy Settings
$DOPPY_API_URL = 'https://assistant.doppy.ir/create';

// ============================================================================
// توابع کمکی
// ============================================================================

function log_request($action, $data = []) {
    $log_entry = [
        'timestamp' => date('Y-m-d H:i:s'),
        'action' => $action,
        'ip' => $_SERVER['REMOTE_ADDR'] ?? '',
        'user_agent' => $_SERVER['HTTP_USER_AGENT'] ?? '',
        'data' => $data
    ];
    
    $log_file = BENZ_LOG_FILE;
    @file_put_contents($log_file, json_encode($log_entry) . PHP_EOL, FILE_APPEND);
}

function send_json($success, $data = [], $message = '') {
    $response = [
        'success' => $success,
        'version' => BENZ_API_VERSION,
        'timestamp' => date('c'),
    ];
    
    if (!$success) {
        $response['error'] = $message;
    } else {
        $response['data'] = $data;
    }
    
    echo json_encode($response, JSON_UNESCAPED_UNICODE);
    exit;
}

function verify_api_key($key, &$key_info) {
    global $VALID_API_KEYS;
    
    if (!isset($VALID_API_KEYS[$key])) {
        return false;
    }
    
    $key_info = $VALID_API_KEYS[$key];
    return true;
}

function get_bearer_token() {
    $headers = getallheaders();
    
    if (isset($headers['Authorization'])) {
        $auth = $headers['Authorization'];
        if (preg_match('/Bearer\s+(.+)/i', $auth, $matches)) {
            return $matches[1];
        }
    }
    
    return '';
}

// ============================================================================
// مسیریابی
// ============================================================================

$request_path = parse_url($_SERVER['REQUEST_URI'], PHP_URL_PATH);
$request_path = str_replace('/api/doppy', '', $request_path);

if (empty($request_path) || $request_path === '/') {
    $request_path = '/test';
}

// ============================================================================
// احراز هویت
// ============================================================================

$api_key = get_bearer_token();
$key_info = [];

if ($request_path !== '/test' && !verify_api_key($api_key, $key_info)) {
    http_response_code(401);
    send_json(false, [], 'Unauthorized - Invalid API Key');
}

// ============================================================================
// Endpoints
// ============================================================================

// GET /test - تست ارتباط
if ($request_path === '/test' && $_SERVER['REQUEST_METHOD'] === 'GET') {
    log_request('test_connection', [
        'client_ip' => $_SERVER['REMOTE_ADDR'] ?? ''
    ]);
    
    send_json(true, [
        'status' => 'online',
        'version' => BENZ_API_VERSION,
        'services' => ['gemini', 'doppy', 'image_download'],
        'timestamp' => date('c')
    ]);
}

// GET /status - وضعیت سرویس
if ($request_path === '/status' && $_SERVER['REQUEST_METHOD'] === 'GET') {
    send_json(true, [
        'status' => 'online',
        'uptime' => 'N/A',
        'services' => [
            'gemini' => 'operational',
            'doppy' => 'operational',
            'image_download' => 'operational'
        ],
        'version' => BENZ_API_VERSION
    ]);
}

// POST /generate/content - تولید محتوا
if ($request_path === '/generate/content' && $_SERVER['REQUEST_METHOD'] === 'POST') {
    $input = json_decode(file_get_contents('php://input'), true);
    
    $keywords = $input['keywords'] ?? '';
    $prompt = $input['prompt'] ?? '';
    $model = $input['model'] ?? 'gemini-2.0-flash';
    $temperature = floatval($input['temperature'] ?? 0.7);
    
    if (empty($keywords)) {
        http_response_code(400);
        send_json(false, [], 'Keywords required');
    }
    
    log_request('generate_content', [
        'keywords' => $keywords,
        'model' => $model
    ]);
    
    // درخواست به Gemini
    $gemini_prompt = empty($prompt) ? "درباره '{$keywords}' یک مقالة حرفه‌ای بنویس" : $prompt;
    
    $payload = [
        'contents' => [
            [
                'parts' => [
                    ['text' => $gemini_prompt]
                ]
            ]
        ],
        'generationConfig' => [
            'temperature' => $temperature,
            'topP' => 0.95,
            'topK' => 64,
            'maxOutputTokens' => 8192,
        ]
    ];
    
    $url = "{$GEMINI_API_URL}/{$model}:generateContent?key={$GEMINI_API_KEY}";
    
    $ch = curl_init($url);
    curl_setopt_array($ch, [
        CURLOPT_RETURNTRANSFER => true,
        CURLOPT_POST => true,
        CURLOPT_POSTFIELDS => json_encode($payload, JSON_UNESCAPED_UNICODE),
        CURLOPT_HTTPHEADER => ['Content-Type: application/json'],
        CURLOPT_TIMEOUT => 60,
        CURLOPT_SSL_VERIFYPEER => false,
    ]);
    
    $response = curl_exec($ch);
    $error = curl_error($ch);
    curl_close($ch);
    
    if ($error) {
        http_response_code(500);
        log_request('generate_content_error', ['error' => $error]);
        send_json(false, [], "Gemini Error: {$error}");
    }
    
    $result = json_decode($response, true);
    
    if (isset($result['error'])) {
        http_response_code(500);
        log_request('generate_content_error', $result['error']);
        send_json(false, [], 'Gemini API Error: ' . ($result['error']['message'] ?? 'Unknown'));
    }
    
    if (!isset($result['candidates'][0]['content']['parts'][0]['text'])) {
        http_response_code(500);
        send_json(false, [], 'Invalid response from Gemini');
    }
    
    $content = $result['candidates'][0]['content']['parts'][0]['text'];
    $parsed = json_decode($content, true);
    
    if (json_last_error() !== JSON_ERROR_NONE) {
        // اگر JSON نبود، خود متن را برگردان
        $parsed = [
            'title' => 'محتوای تولید شده',
            'content' => $content,
            'metaDescription' => substr($content, 0, 150),
            'keyphrase' => $keywords,
            'imagePrompt' => "Professional illustration about {$keywords}",
            'imageAlt' => "تصویر مرتبط با {$keywords}",
            'keywords' => $keywords,
        ];
    }
    
    send_json(true, $parsed);
}

// POST /generate/image - تولید تصویر
if ($request_path === '/generate/image' && $_SERVER['REQUEST_METHOD'] === 'POST') {
    $input = json_decode(file_get_contents('php://input'), true);
    
    $prompt = $input['prompt'] ?? '';
    $quality = $input['quality'] ?? 'high';
    
    if (empty($prompt)) {
        http_response_code(400);
        send_json(false, [], 'Prompt required');
    }
    
    log_request('generate_image', ['prompt' => substr($prompt, 0, 100)]);
    
    // درخواست به Doppy
    $url = $DOPPY_API_URL . '?p=' . urlencode($prompt);
    
    $ch = curl_init($url);
    curl_setopt_array($ch, [
        CURLOPT_RETURNTRANSFER => true,
        CURLOPT_TIMEOUT => 60,
        CURLOPT_SSL_VERIFYPEER => false,
    ]);
    
    $response = curl_exec($ch);
    $error = curl_error($ch);
    curl_close($ch);
    
    if ($error) {
        http_response_code(500);
        log_request('generate_image_error', ['error' => $error]);
        send_json(false, [], "Doppy Error: {$error}");
    }
    
    $result = json_decode($response, true);
    
    if (isset($result['error'])) {
        http_response_code(500);
        send_json(false, [], 'Doppy API Error');
    }
    
    // پشتیبانی از فرمت‌های مختلف
    $image_url = $result['image_url'] ?? $result['url'] ?? $result['images'][0] ?? null;
    
    if (empty($image_url)) {
        http_response_code(500);
        send_json(false, [], 'No image URL in response');
    }
    
    send_json(true, [
        'image_url' => $image_url,
        'size' => $result['file_size'] ?? 'unknown',
        'generation_time' => $result['generation_time'] ?? 'N/A'
    ]);
}

// POST /file/download - دانلود تصویر
if ($request_path === '/file/download' && $_SERVER['REQUEST_METHOD'] === 'POST') {
    $input = json_decode(file_get_contents('php://input'), true);
    
    $image_url = $input['image_url'] ?? '';
    
    if (empty($image_url)) {
        http_response_code(400);
        send_json(false, [], 'Image URL required');
    }
    
    log_request('file_download', ['url' => $image_url]);
    
    // دانلود تصویر
    $ch = curl_init($image_url);
    curl_setopt_array($ch, [
        CURLOPT_RETURNTRANSFER => true,
        CURLOPT_TIMEOUT => 30,
        CURLOPT_SSL_VERIFYPEER => false,
    ]);
    
    $image_data = curl_exec($ch);
    $error = curl_error($ch);
    curl_close($ch);
    
    if ($error) {
        http_response_code(500);
        send_json(false, [], "Download Error: {$error}");
    }
    
    // ذخیره موقتی
    $filename = 'doppy_' . time() . '.png';
    $filepath = BENZ_CACHE_DIR . $filename;
    
    file_put_contents($filepath, $image_data);
    
    send_json(true, [
        'file_url' => 'https://pr.benzinyab.com/api/doppy/file/get?file=' . $filename,
        'filename' => $filename,
        'size' => strlen($image_data)
    ]);
}

// GET /file/get - دریافت فایل
if ($request_path === '/file/get' && $_SERVER['REQUEST_METHOD'] === 'GET') {
    $filename = $_GET['file'] ?? '';
    
    if (empty($filename) || strpos($filename, '..') !== false) {
        http_response_code(400);
        die('Invalid filename');
    }
    
    $filepath = BENZ_CACHE_DIR . basename($filename);
    
    if (!file_exists($filepath)) {
        http_response_code(404);
        die('File not found');
    }
    
    header('Content-Type: image/png');
    header('Content-Length: ' . filesize($filepath));
    readfile($filepath);
    exit;
}

// POST /analytics/sync - همگام‌سازی آمار
if ($request_path === '/analytics/sync' && $_SERVER['REQUEST_METHOD'] === 'POST') {
    $input = json_decode(file_get_contents('php://input'), true);
    
    log_request('analytics_sync', $input);
    
    send_json(true, [
        'status' => 'synced',
        'timestamp' => date('c')
    ]);
}

// 404 - مسیر یافت نشد
http_response_code(404);
send_json(false, [], 'Endpoint not found');
